
###########################
#parseToNodes
###########################
flowControlFunctions <- c("if","while","repeat","for",
	#rest is special
	"function"
	)
	
deparseTreeExp <- function(e){
	s <- NULL
	d <- deparse(e, width.cutoff = 400)
	if(length(d) > 1){
		for(x in d){
			s <- paste(s, x, "\n" ,sep="")
		}
		s <- substring(s, 1, nchar(s, type="chars")-1)
		return(s)
	}
	return(d)
}

parseToNodes <- function(text = NULL, file = "", keep.source=getOption("keep.source")){
	if (is.null(text) & (is.null(file) || nchar(file) == 0)) {
		return(list())
	}
	
	#save old keep.source option
	old.keep.source <- getOption("keep.source")
	#restore options on exit
	on.exit(options(keep.source=old.keep.source))
	#change keep.source option
	options(keep.source=keep.source)
	
	#"parse" uses keep.souce option
	codes <- parse(text = text, file = file)
	
	
	lines <- list()
	
	if (length(codes) > 0) {
		for(i in 1:length(codes)){
			e <- codes[[i]]
			flist <- retrieveFunction(e)
			lines[[i]] <- flist
		}
	}
	
	if(!keep.source){
		return(lines)
	}
	
	#if keep.source, add non code lines
	lines2 <- list()
	ref <- attr(codes, "srcref", exact=TRUE)
	wref <- attr(codes, "wholeSrcref", exact=TRUE)
	src <- attr(codes, "srcfile", exact=TRUE)
	if (is.null(ref) || is.null(wref) || is.null(src)) {
		return(lines)
	}
	
	codeIndex <- 1
	linesIndex <- 1
	lineNum <- wref[1]
	while(lineNum <= wref[3]){
		#skip last empty line
		if(lineNum == wref[3] && wref[4] == 0){
			break
		}
		if(codeIndex > length(ref) || lineNum < ref[[codeIndex]][1]){
			#comment or empty lines
			lines2[[linesIndex]] <- list(line=getSrcLines(src, lineNum, lineNum)[1])
			lines2[[linesIndex]]$srcref <- as.integer(c(lineNum-1, lineNum-1, 0, -1))
			linesIndex <- linesIndex + 1
			lineNum <- lineNum + 1
		}else{
			#code line
			while(codeIndex <= length(ref) && lineNum == ref[[codeIndex]][1]){
				lines2[[linesIndex]] <- lines[[codeIndex]]
				
				#srcLines
				srcLines <- as.character(ref[[codeIndex]])
				
				#if next code is not in the same line, replace last line to whole line
				#to include line tail comment
				if (!(length(ref) > codeIndex && ref[[codeIndex+1]][1]==ref[[codeIndex]][3])){
					#lastLine <- getSrcLines(src, ref[[codeIndex]][3], ref[[codeIndex]][3])[1]
					#srcLines[length(srcLines)] <- lastLine
					refTemp <- ref[[codeIndex]]
					refTemp[c(4,6)] <- .Machine$integer.max
					srcLines <- as.character(refTemp)
				}
				
				#joint lines srcLines to srcLine
				srcLine <- ""
				for(x in srcLines){
					srcLine <- paste(srcLine, x, "\n", sep="")
				}
				srcLine <- substring(srcLine, 1, nchar(srcLine, type="chars")-1)
				
				lines2[[linesIndex]]$srcline <- srcLine
				lines2[[linesIndex]]$srcref <- ref[[codeIndex]][c(1,3,5,6)]-as.integer(1)
				
				lineNum <- ref[[codeIndex]][3]
				codeIndex <- codeIndex + 1
				linesIndex <- linesIndex + 1
			}
			lineNum <- lineNum + 1
		}
	}
	
	return(lines2)
}


retrieveFunction <- function(e){
	env <- new.env()
	
	env$line = deparseTreeExp(e)
	
	functionCollect <- function(v, e){
		assign(v, e, envir = env)
	}
	w <- functionWalker(functionCollect)
	w$top = TRUE
	
	codetools::walkCode(e, w)
	
	as.list(env)
}

functionHandler <- function(v, w){
	switch(v, "="=,"<-"=assignHandler)
}

assignHandler <- function(e, w) {
	w$collect("symbol", deparseTreeExp(e[[2]]))
	w$top = FALSE
	codetools::walkCode(e[[3]], w)
}

functionCall <- function(e, w){
	w$collect("sentence", deparseTreeExp(e))
	fun <- deparseTreeExp(e[[1]])
	if(fun %in% flowControlFunctions){
		w$collect("control", TRUE)
	}else{
		w$collect("func", fun)
		args <- as.list(e[-1])
		args <- lapply(args, function(x){x <- deparseTreeExp(x)})
		w$collect("args", args)
	}
}

functionLeaf <- function(e, w){
	w$collect("sentence", deparseTreeExp(e))
}

functionWalker <- function(collect){
	w <- codetools::makeCodeWalker(handler = functionHandler, call = functionCall, leaf = functionLeaf)
	w$collect = collect
	w
}

.rflowFindNewVariables <- function(text = NULL, file = "") {
	exp = parse(file = file, text = text)
	v <- sapply(exp, function(x){
		if (length(x) >= 3 && (x[[1]] == "<-" || x[[1]] == "="))
			deparseTreeExp(x[[2]])
		else
			NA
	})
	v[!is.na(v)]
}

###########################
#r -> java function
###########################

.rflowRJavaInterfaceClass <- NULL

.rflow.pager <- function(file, header, title, delete.file) {
	invisible(rJava::.jcall(.rflowRJavaInterfaceClass, "V", "pager", as.character(file), as.character(header), as.character(title), as.logical(delete.file)))
}

.rflow.browser <- function(url) {
	## check if R help
	isRHelp <- FALSE
	m <- regexec("^(([^:]+)://)?([^:/]+)(:([0-9]+))?(/.*)", url)
	result <- regmatches(url, m)
	if (length(result) > 0 && length(result[[1]]) >= 6) {
		host <- result[[1]][4]
		port <- result[[1]][6]
		# httpdPort is not public variable, so this may fail when updated.
		isHelpPort <- try(if (is.function(tools:::httpdPort)) {
			port == tools:::httpdPort() # R >= 3.2.0
		} else {
			port == tools:::httpdPort # R < 3.2.0 
		}, silent = TRUE)
		isRHelp <- is.logical(isHelpPort) && isHelpPort && grepl("^127", host)
	}
	
	if (isRHelp) {
		invisible(rJava::.jcall(.rflowRJavaInterfaceClass, "V", "browser", url, isRHelp))
	} else {
		r_browser <- Sys.getenv("R_BROWSER")
		if (!nzchar(r_browser)) {
			r_browser <- get(".original.browser", envir = parent.frame())
		}
		browseURL(url, browser = r_browser)
	}
}

.rflow.flush.console <- function() {
	invisible(rJava::.jcall(.rflowRJavaInterfaceClass, "V", "flushConsole"))
}

.rflowPackageAttached <- function(pkgname, ...) {
	invisible(rJava::.jcall(.rflowRJavaInterfaceClass, "V", "packageStatus", pkgname, TRUE))
}

.rflowPackageDetached <- function(pkgname, ...) {
	invisible(rJava::.jcall(.rflowRJavaInterfaceClass, "V", "packageStatus", pkgname, FALSE))
}

.rflow.view <- function(x) {
	command <- deparse(substitute(x), control = "keepNA", nlines = 1)
	id <- sprintf("v%d", get(".rflowViewIndex", envir = .RflowEnv))
	count <- 0
	while (exists(id, envir = .RflowEnv$.VIEW_ENV)) {
		.RflowEnv$.rflowViewIndex <- .RflowEnv$.rflowViewIndex + 1
		if (.RflowEnv$.rflowViewIndex > 10^10) {
			.RflowEnv$.rflowViewIndex <- 0
		}
		
		# to avoid infinite loop
		count <- count + 1
		if (count > 10000) {
			return()
		}
		
		id <- sprintf("v%d", .RflowEnv$.rflowViewIndex)
	}
	assign(id, x, envir = .RflowEnv$.VIEW_ENV)
	invisible(rJava::.jcall(.rflowRJavaInterfaceClass, "V", "view", command, id))
}

View <- function(x) {
	cl <- match.call()
	
	# use utils::View if .USE_RFLOW_VIEW is not true
	cl[[1]] <- if(exists(".USE_RFLOW_VIEW", envir = .RflowEnv) && !is.null(.RflowEnv$.USE_RFLOW_VIEW) && .RflowEnv$.USE_RFLOW_VIEW) {
		quote(.RflowEnv$.rflow.view)
	} else {
		quote(utils::View)
	}
	
	eval.parent(cl)
}

###########################
#Rflow function
###########################

#RflowGD
#In JavaGD, we can't get the device number until plot begins. So always do plot.new.
RflowGD <- function(width = NULL, height = NULL, ps = NULL, ...){
	if ( is.null(width) || is.null(height) ){
		size <- get(".RFLOW_GD_SIZE", envir = parent.frame())
		width <- size[1]
		height <- size[2]
		if (require("rJava")) {
			# create device UI beforehand to get the canvas size
			size <- rJava::.jcall(Sys.getenv("JAVAGD_CLASS_NAME"), "[I", "openDevice", width, height)
			width <- size[1]
			height <- size[2]
		}
	}
	if ( is.null(ps) ) {
		ps <- get(".RFLOW_GD_POINT_SIZE", envir = parent.frame())
	}
	
	if(.Platform$OS.type == "windows" && R.version$major >= 4) {
		# Workaround to handle non-ASCII class paths.
		# May not work on recent Java (e.g. Java 16)
		if ("rJava" %in% .packages() && !identical(JavaGD:::.javaGD.get.class.path(), "")) {
			cp <- JavaGD:::.javaGD.get.class.path()
			Encoding(cp) <- "UTF-8"
			cp <- unique(unlist(strsplit(cp, .Platform$path.sep)))
			rJava::.jaddClassPath(cp)
			JavaGD:::.javaGD.set.class.path("")
		}
		invisible(.Call(JavaGD:::newJavaGD, name="JavaGD", width=width, height=height, ps=ps))
	} else {
		JavaGD::JavaGD(width=width, height=height, ps=ps, ...)
	}

	
	par(bg = "transparent")
	plot.new()
}


#clear all objects of envir
.clearObjects <- function(envir = .GlobalEnv)
{
	rm(list = ls(envir = envir, all.names = TRUE), envir = envir)
}

#new graphics device
.newGraphics <- function(...)
{
	do.call(getOption("device"), list(...))
}

#get function arguments
.rflowGetFormals <- function (fun)
{
	if (is.character(fun)) {
		fun <- get(fun, pos=1)
	}
	if (is.primitive(fun)) {
		formals <- formals(args(fun))
	} else {
		formals <- formals(fun)
	}
	for(i in 1:length(formals))
	{
		formals[[i]] <- deparse(formals[[i]])
	}
	return(formals)
}

#get function's local variables
.rflowFindLocals <- function (fun, merge = FALSE) 
{
    vars <- new.env(TRUE)
    funs <- new.env(TRUE)
    enter <- function(type, v, e, w) if (type == "function") 
        assign(v, TRUE, funs)
    else assign(v, TRUE, vars)
    collectUsage(fun, enterLocal = enter)
    fnames <- ls(funs, all.names = TRUE)
    vnames <- ls(vars, all.names = TRUE)
    if (merge) 
        sort(unique(c(vnames, fnames)))
    else list(functions = fnames, variables = vnames)
}

#command support
.rflowSupportCommand <- function(x)
{
	ret <- NULL
	packages <- NULL
	
	# namespace
	# e.g. rpart::rpart
	if ((a <- regexpr(':::?', x)) > 1) {
		ns <- substr(x, 1, a-1)
		match.length <- attr(a, "match.length")
		if (match.length == 3) {
			# :::
			ret <- ls(envir=getNamespace(ns), all.names = TRUE)
		} else {
			# ::
			ret <- getNamespaceExports(ns = ns)
		}
		
		# datasets
		# getNamespaceInfo() is not public function, so use safely
		datasets <- tryCatch(suppressWarnings(ls(getNamespaceInfo(ns, "lazydata"), all.names=TRUE)), error=function(e){})
		if (length(datasets) > 0) {
			ret <- c(ret, datasets)
		}
		
		# filter result
		x <- substring(x, a + match.length)
		if (nzchar(x)) {
			pat <- gsub("\\.", "\\\\.", x)
			pat <- paste("^", pat, ".*", sep="")
			ret <- grep(pat, ret, ignore.case=TRUE, value=TRUE)
		}
		return(list(word=ret, namespace=ns))
	}
	
	
	#get environment names
	#remove preceding 'package:'
	searchNames <- search()
	searchNum <- length(searchNames)
	
	if ((a <- regexpr('\\$', x)) > 1){
		#e.g. 'iris$' 'Boston$c'
		b <- regexpr("\\$[^\\$]*$", x)
		multipleDepth <- a != b
		aList <- substr(x, 1, a-1)
		bList <- if (multipleDepth) substr(x, 1, b-1) else aList
		isFilter <- a < nchar(x)
		if (isFilter) {
			y <- substr(x, b+1, nchar(x))
			pat <- paste('^`?', y, sep="")
			#escape dot
			pat <- gsub("\\.", "\\\\.", pat)
		}
		
		for (i in 1:searchNum) {
			if (exists(aList, where = i, inherits = FALSE)) {
				obj <- if (multipleDepth) {
					eval(parse(text=bList), envir = as.environment(searchNames[i]))
				} else {
					get(aList, pos = i, inherits = FALSE)
				}
				listNames <- .rflowGetNames(obj, check.names=TRUE)
				if (isFilter) {
					listNames <- listNames[grep(pat, listNames, ignore.case=TRUE)]
				}
				ret <- c(ret, listNames)
				packages <- c(packages, rep(i, length(listNames)))
			}
		}
	} else if ((a <- regexpr('@', x)) > 1){
		#e.g. 'iris@' 'Boston@c'
		b <- regexpr("@[^@]*$", x)
		multipleDepth <- a != b
		aObj <- substr(x, 1, a-1)
		bObj <- if (multipleDepth) substr(x, 1, b-1) else aObj
		isFilter <- a < nchar(x)
		if (isFilter) {
			y <- substr(x, b+1, nchar(x))
			pat <- paste('^`?', y, sep="")
			#escape dot
			pat <- gsub("\\.", "\\\\.", pat)
		}
		
		for (i in 1:searchNum) {
			if (exists(aObj, where = i, inherits = FALSE)) {
				obj <- if (multipleDepth) {
					eval(parse(text=bObj), envir = as.environment(searchNames[i]))
				} else {
					get(aObj, pos = i, inherits = FALSE)
				}
				if (!isS4(obj)) {
					next
				}
				slotNamez <- slotNames(obj)
				if (isFilter) {
					slotNamez <- slotNamez[grep(pat, slotNamez, ignore.case=TRUE)]
				}
				ret <- c(ret, slotNamez)
				packages <- c(packages, rep(i, length(slotNamez)))
			}
		}
	} else if (!nzchar(x)) {
		# if empty, return global objects list
		ret <- ls(pos=1L, all.names=TRUE)
		packages <- rep(1L, length(ret))
	} else {
		#escape dot
		x <- gsub("\\.", "\\\\.", x)
		pat <- paste("^", x, ".*", sep="")
		for(i in 1:searchNum) {
			namez <- grep(pat, ls(pos=i, all.names=TRUE), ignore.case=TRUE, value=TRUE)
			ret <- c(ret, namez)
			packages <- c(packages, rep(i, length(namez)))
		}
	}
	
	if (is.null(ret)) {
		return(NULL)
	}else{
		return(list(word=ret, package=packages, search=searchNames))
	}
}


#get objects filtered by FILTERFUN
#FILTERFUN must return TRUE/FALSE
.rflowGetObjectsFiltered <- function(envir = .GlobalEnv, FILTERFUN)
{
	a <- unlist(eapply(envir, FILTERFUN))
	if(!is.null(a)){
		a <- sort(names(which(a)))
	}
	a
}

#get length or rownum of obj
.rflowGetLength <- function(obj)
{
	tip <- dim(obj)
	if(is.null(tip)){
		tip <- length(obj)
		#if(len == 1){
		#	tip <- NULL
		#}else{
		#	tip <- len
		#}
	}else{
		tip <- tip[1]
	}
	
	return(tip)
}

#get help rd object
#topic: topic name of help
#filter: character vector of Rtag name. NULL for no filter
.rflowGetRd <- function(topic){
	rd <- utils:::.getHelpFile(help(topic))
	rd
}

.rflowGetArgHelps <- function(topic){
	tools:::.Rd_get_argument_table(.rflowGetRd(topic))
}

#get help html
#topic: topic name of help
#dynamic: html links if TRUE
.rflowGetHtmlHelp <- function(topic, dynamic=FALSE, stylesheet=NULL){
	helpDir <- eval(parse(text=paste("?", topic, sep="")))
	helpDir <- helpDir[length(helpDir)]
	rd <- utils:::.getHelpFile(helpDir)
	html <- character()
	htmlCon <- textConnection("html", open="w", local=TRUE)
	on.exit(close(htmlCon))
	if (is.null(stylesheet)) {
		stylesheet <- file.path(R.home("doc"), "html", "R.css")
	}
	tools::Rd2HTML(Rd=rd, out=htmlCon, package=dirname(dirname(helpDir)),
		dynamic=dynamic, stylesheet=stylesheet, outputEncoding=localeToCharset())
	html
}

.rflowHelp <- function(pattern) {
	# exact match
	result <- help(topic = pattern, try.all.packages = TRUE)
	if (length(result) > 0) {
		return(result)
	}
	# search
	help.search(pattern = pattern)
}

# Gets R object info list.
# Internal function called from .rflowGetObject() and .rflowGetObjects().
#
# obj: The object which get info from
# name: character (or numeric). character representation of 'obj'. Used to check if name is valid.
#       Invalid examples) '124abc', 'a b c'
# notExpression: logical. Check if 'name' is expression if FALSE. Used for 'name' validity.
#       Expression example) a$b
#       Non-expression example) iris
.rflowObjectInfo <- function(obj, name, notExpression = TRUE) {
	objInfo <- list()
	#check if obj is valid
	type <- NULL
	try(suppressWarnings(type <- typeof(obj)), silent=TRUE)
	if(is.null(type)){
		return(objInfo)
	}
	objInfo$type <- typeof(obj)
	objInfo$class <- class(obj)
	objInfo$newclass <- is(obj)
	
	#flag
	flag <- 0
	if(is.data.frame(obj)){flag <- flag+1}
	if(is.matrix(obj)){flag <- flag+2}
	if(is.array(obj)){flag <- flag+2^2}
	objInfo$flag <- as.integer(flag)
	
	# extended information
	no.dim <- FALSE
	if (!is.null(.RFLOW_EXTENDED_INFO_FUNCTIONS) && length(.RFLOW_EXTENDED_INFO_FUNCTIONS) > 0) {
		exInfo <- new.env()
		lapply(.RFLOW_EXTENDED_INFO_FUNCTIONS, function(f) {
			do.call(what = f, args = list(obj = obj, objInfo = objInfo, exInfo = exInfo))
		})
		objInfo$exInfo <- as.list(exInfo)
		if ("no.dim" %in% names(objInfo$exInfo)) {
			no.dim <- objInfo$exInfo$no.dim
		}
	}
	
	dimu <- 0L
	tip <- ""
	if (!no.dim) {
		dimu <- dim(obj)
		if (is.numeric(dimu) && !is.integer(dimu)) {
			dimu <- as.integer(dimu)
		}
		if (is.null(dimu)){
			if (type == "S4") {
				dimu <- length(slotNames(obj))
			} else {
				len <- length(obj)
				if (len == 1 && !is.list(obj)){
					tip <- deparse(obj, width.cutoff = 100, nlines = 1)
					dimu <- as.integer(1)
				} else {
					dimu <- as.integer(len)
				}
			}
		}
	}
	objInfo$dim <- dimu
	objInfo$tip <- tip
	if (notExpression || missing(name) || is.null(name)) {
		isExpression <- FALSE
	} else {
		isExpression <- length(parse(text=name)[[1]]) > 1
	}
	objInfo$validname <- ifelse(isExpression | is.numeric(name), TRUE, (make.names(name)==name))
	objInfo$pointer <- .rflowGetPointer(obj)
	objInfo
}

.rflowGetObject <- function(parent = .GlobalEnv, obj=NULL, name, fullName){
	if (is.null(obj)){
		obj <- get(fullName, envir=parent)
	}
	#if (is.character(obj)) {
	#	obj <- get(obj, envir=parent)
	#}
	objInfo <- .rflowObjectInfo(obj, name, FALSE)
	if (length(objInfo) < 1) {
		return(NULL)
	}
	objInfo
}
	
	
#get objects information
#all.names: see ls
#limit: maximum of obj infos to be returned
.rflowGetObjects <- function(parent = .GlobalEnv, all.names = FALSE, limit = 200, offset = 0, depth = 0)
{
	objects <- list()
	#attr name
	overlimit <- "overlimit"
	
	getObjInfo <- function(name, obj, depth) {
		objInfo <- .rflowObjectInfo(obj, name, TRUE)
		# children
		if (depth > 0 && !is.atomic(obj)) {
			objInfo$children <- .rflowGetObjects(parent = obj, all.names = all.names, limit = limit, offset = 0, depth = depth-1)
		}
		objInfo
	}
	
	parentType <- NULL
	try(parentType <- typeof(parent), silent=TRUE)
	if(is.null(parentType)){
		return(objects)
	}
	if(parentType == "environment"){
		objs <- ls(parent, all.names = all.names)
		if(length(objs) > 0){
			#len <- ifelse(limit > 0 && length(objs) > limit, limit, length(objs))
			len <- length(objs) - offset
			if (limit > 0)
				len <- min(len, limit)
			objects <- vector("list", len)
			for(i in seq_len(len)){
				objects[[i]] <- getObjInfo(objs[i+offset], parent[[objs[i+offset]]], depth)
				names(objects)[i] <- objs[i+offset]
			}
			attr(objects, overlimit) <- length(objs)
		}
	}else if(parentType == "list" || parentType == "pairlist"){
		namez <- names(parent)
		if(is.null(namez)){
			#no names
			if(length(parent) > 0){
				#len <- ifelse(limit > 0 && length(parent) > limit, limit, length(parent))
				len <- length(parent) - offset
				if (limit > 0)
					len <- min(len, limit)
				objects <- vector("list", len)
				for(i in seq_len(len)){
					objects[[i]] <- getObjInfo(i+offset, parent[[i+offset]], depth)
					#names(objects)[i] <- i
				}
				#if(length(parent) > limit){
					attr(objects, overlimit) <- length(parent)
				#}
			}
		}else{
			if(length(namez) > 0){
				#len <- ifelse(limit > 0 && length(namez) > limit, limit, length(namez))
				len <- length(namez) - offset
				if (limit > 0)
					len <- min(len, limit)
				objects <- vector("list", len)
				for(i in seq_len(len)){
					#empty name is index
					x <- ifelse(namez[i+offset] == "", i+offset, namez[i+offset])
					objects[[i]] <- getObjInfo(x, parent[[i+offset]], depth)
					names(objects)[i] <- namez[i+offset]
				}
				#if(length(namez) > limit){
					attr(objects, overlimit) <- length(namez)
				#}
			}
		}
	} else if (parentType == "S4") {
		namez <- slotNames(parent)
		if (length(namez) > 0) {
			len <- length(namez) - offset
			if (limit > 0)
				len <- min(len, limit)
			for(i in seq_len(len)){
				objects[[i]] <- getObjInfo(namez[i+offset], slot(parent, namez[i+offset]), depth)
				names(objects)[i] <- namez[i+offset]
			}
			attr(objects, overlimit) <- length(namez)
		}
	}
	
	return(objects)
}

# levels
.rflowLevels <- function(x, n=100000) {
	if (is.list(x)) {
		return(unique(unlist(lapply(x, function(v) {.rflowLevels(v, n)}))))
	}
	if (is.factor(x)) {
		values <- levels(x)
	} else {
		values <- levels(as.factor(head(x, n=n)))
	}
	if (is.character(x) || is.factor(x)) {
		values <- encodeString(values, quote='"')
	}
	values
}

# default CRAN mirror
.rflowDefaultCRANMirror <- function() {
	mirrors <- getCRANmirrors(local.only = TRUE)
	if (is.data.frame(mirrors) && nrow(mirrors) > 0) {
		return(mirrors[1, "URL"])
	}
	return(NULL)
}

# replace CRAN mirror
.rflowReplaceCRANMirror <- function(url = "@CRAN@") {
	r <- getOption("repos")
	r["CRAN"] <- gsub("/$", "", url)
	options(repos = r)
}

# rank generic methods
.rflowMethodRank <- function(classes, methods) {
	ret1 <- character()
	ret2 <- character()
	for (method in methods) {
		hit <- FALSE
		for (class in classes) {
			# getS3method() must be called in global env to search every methods
			if (!is.null(eval(substitute(getS3method(method, class, optional = TRUE), list(method=method, class=class)), envir = .GlobalEnv))) {
				hit <- TRUE
				break;
			}
		}
		if (hit) {
			ret1 <- append(ret1, method)
		} else {
			ret2 <- append(ret2, method)
		}
	}
	return(c(ret1, ret2))
}

# factor like columns
.rflowFactorLikeColumns <- function(x, check.names = FALSE, n = 100, levels = 20) {
	namez <- names(x)
	if (is.null(namez)) {
		return(NULL)
	}
	if (check.names) {
		namez = sapply(namez, 
			function(x){if (make.names(x) == x) x else encodeString(x = x, quote = "`")},
			USE.NAMES = FALSE)
	}
	if (is.list(x)) {
		counts <- sapply(head(x, n), function(column){length(unique(column))})
	} else {
		counts <- apply(head(x, n), 2, function(column){length(unique(column))})
	}
	#namez[counts <= levels]
	namez[order(counts)]
}

# get names
.rflowGetNames <- function(x, check.names = FALSE, filter) {
	namez <- names(x)
	if (is.null(namez)) {
		return(NULL)
	}
	if (check.names) {
		namez = sapply(namez, 
			function(x){if (make.names(x) == x) x else encodeString(x = x, quote = "`")},
			USE.NAMES = FALSE)
	}
	if (missing(filter)) {
		return(namez)
	}
	hit <- sapply(x, filter)
	if (is.vector(hit))
		namez[hit]
	else
		namez
}

# binary variables
.rflowBinaryVariables <- function(x) {
	na.omit(sapply(names(x), function(name){
		y <- x[[name]]
		if (is.logical(y) || all(na.omit(y) %in% c(0, 1))) {
			if (name != make.names(name)) {
				name = encodeString(name, quote="`")
			}
			name
		} else {
			NA
		}
	}))
}

# for loop elements
.rflowForElements <- function(loopChar) {
	exp <- parse(text=paste("for(", loopChar, "){}", sep=""))[[1]]
	expToChar <- function(e) {
		paste(deparse(e), sep="", collapse = "")
	}
	c(expToChar(exp[[2]]), expToChar(exp[[3]]))
}

# No warning, debug or trace.
# Return try-error characters if error occured.
.rflowSilent <- function(expr, debugState = TRUE) {
	if (debugState) {
		.rflowNoDebug(suppressWarnings(try(expr, silent = TRUE)))
	} else {
		.rflowNoTracing(suppressWarnings(try(expr, silent = TRUE)))
	}
}

# set debug and trace off, put it back in the end
.rflowNoDebug <- function(expr) {
	oldDebuggingState <- debuggingState(on = FALSE)
	oldTracingState <- tracingState(on = FALSE)
	on.exit({
		debuggingState(on=oldDebuggingState)
		tracingState(on=oldTracingState)
	})
	evalq(expr)
}

.rflowNoTracing <- function(expr) {
    oldTracingState <- tracingState(on = FALSE)
    on.exit({
        tracingState(on=oldTracingState)
    })
    evalq(expr)
}

.rflowCheckSyntax <- function(text, debugState = TRUE) {
  if (debugState) {
    .rflowNoDebug(suppressWarnings(tryCatch(is.expression(parse(text=text)), error=function(e){conditionMessage(e)})))
  } else {
    .rflowNoTracing(suppressWarnings(tryCatch(is.expression(parse(text=text)), error=function(e){conditionMessage(e)})))
  }
}

#check whether in browse or not
#if in browse, .rflowBrowserObject is value of browserText(), otherwise NULL
#if in browse, .rflowDebugEnvironment is set
#if in browse, .rflowCurrentCall is set
.rflowCheckBrowser <- function(envir){
	tryCatch(assign(".rflowBrowserObject", browserText(), envir=envir), error=function(e){assign(".rflowBrowserObject", NULL, envir=envir)})
	brObj <- get(".rflowBrowserObject", envir=envir)
	if (!is.null(brObj)){
		# expecting called as '.rflowSilent(.rflowCheckBrowser())'
		# this is 12 frames
		index <- -1:-12
		if (brObj==".rflowRecover"){
			# if called while recover add 3 frames
			index <- -1:-15
		}
		
		frames <- sys.frames()
		try(assign(".rflowDebugEnvironment", rev(frames)[index], envir=envir), silent=TRUE)
		
		calls <- sys.calls()
		try(assign(".rflowCurrentCall", lapply(rev(calls)[index], format), envir=envir), silent=TRUE)
		
	} else {
		try(assign(".rflowDebugEnvironment", list(globalenv())), silent=TRUE)
	}
	invisible()
}

.rflowAssignInViewEnv <- function(x) {
     id <- sprintf("v%d", get(".rflowViewIndex", envir = .RflowEnv))
     count <- 0
     while (exists(id, envir = .RflowEnv$.VIEW_ENV)) {
         .RflowEnv$.rflowViewIndex <- .RflowEnv$.rflowViewIndex + 1
         if (.RflowEnv$.rflowViewIndex > 10^10) {
             .RflowEnv$.rflowViewIndex <- 0
         }
         
         # to avoid infinite loop
         count <- count + 1
         if (count > 10000) {
             return(NULL)
         }
         
         id <- sprintf("v%d", .RflowEnv$.rflowViewIndex)
     }
     assign(id, x, envir = .RflowEnv$.VIEW_ENV)
     return(id)
}

.rflowGetPointer <- function(x) {
	tryCatch(deparse(.Internal(address(x)), width.cutoff = 200L), error = function(e) { NULL })
}

.rflowGetConnectionInfo <- function(con) {
	if (is.character(con)) {
		list(description = con, class = "file")
	} else if (inherits(con, "connection")) {
		s <- summary(con)
		list(description = s$description, class = s$class)
	} else {
		NULL
	}
}

# Add a function which add extended information to R object info in .rflowGetObjects() and .rflowGetObject()
# f: This function must have the following arguments
# function(obj, objInfo, exInfo, ...) { }
#   obj: The R object from which you get information.
#   objInfo: The list of R object infomation that have already collected. (e.g. objInfo$type, objInfo$class, ...)
#   exInfo: The environment object which you can add information to.
#
# You can add info by adding new variable to 'exInfo'.
# Variable names bellow are reserved and have a specific perpose.
#   no.dim: logical. if TRUE, skip dim calculation. Used when dim() is slow.
#
# e.g.) function(obj, objInfo, exInfo, ...) {
#	if (inherits(obj, "slow.dim")) { exInfo$no.dim <- TRUE }
# 	if (is.data.frame(obj)) { exInfo$my.isDataFrame <- TRUE }
# 	if (is.matrix(obj)) { exInfo$my.type <- "matrix" }
# }
.rflowAddExtendedInfoFunction <- function(f) {
	if (is.function(f)) {
		.RflowEnv$.RFLOW_EXTENDED_INFO_FUNCTIONS <- c(.RflowEnv$.RFLOW_EXTENDED_INFO_FUNCTIONS, f)
	}
}

# Evaluate objects in 'env' to ensure all promise objects are evaluated
.rflowForceEvaluation <- function(env, all.names = FALSE) {
	objNames <- ls(envir = env, all.names = all.names)
	invisible(eval(parse(text=paste("force(", objNames, ")", sep = "", collapse = ";")), envir = env))
}


.UNDO_ENV <- new.env()

#variable for error check
.rflowErrorObject <- NULL

#varibale for browser check
.rflowBrowserObject <- NULL

#variable for refering debug environment
.rflowDebugEnvironment <- list(globalenv())

#variable for refering current call
.rflowCurrentCall <- NULL

#variable for using recover
.rflowRecover <- FALSE

#variable for debug
.RFLOW_DEBUG <- FALSE

#variable for graphics device size
#7inch in 72dpi = 504px
#7inch in 96dpi = 672px
.RFLOW_GD_SIZE <- c(600, 600)
.RFLOW_GD_POINT_SIZE <- 12

#preview environment
.PREVIEW_ENV <- new.env()

# object for View
.VIEW_ENV <- new.env()
.rflowViewIndex <- 0

.original.browser <- NULL

.rflowFrame <- NULL

# List of functions which add extended information to R object info in .rflowGetObjects() and .rflowGetObject()
.RFLOW_EXTENDED_INFO_FUNCTIONS <- list()

#-------------------------------------------
# 
#-------------------------------------------

# export R 'object' to 'folder'
.exportObject <- function(object, folder) {
	
	# object.RData
	save(object, file = paste(folder, "/object.RData", sep = ""))
	
	
	# text connection 
	createConnection <- function(fileName) {
		file(description = paste(folder, "/", fileName, sep = ""), open = "w", encoding = "UTF-8")
	}

	# summary.txt
	writeSummary <- function() {
		con <- createConnection("summary.txt")
		sink(file = con)
		on.exit(sink(), add = TRUE)
		on.exit(close(con), add = TRUE)
		
		print(summary(object))
	}

	# head.txt
	writeHead <- function() {
		con <- createConnection("head.txt")
		on.exit(close(con), add = TRUE)
		
		if (inherits(object, c("matrix", "data.frame"))) {
			write.table(head(object, n = 100), file = con, sep = "\t")
		} else {
			write(head(object, n = 100), file = con, sep = "\n")
		}
	}

	# dim.txt
	writeDim <- function() {
		con <- createConnection("dim.txt")
		on.exit(close(con), add = TRUE)

		x = dim(object)
		if (is.null(x))
			x = length(object)
		write(x, file = con, sep = "\n")		
	}

	# ls.txt
	writeLs <- function() {
		con <- createConnection("ls.txt")
		on.exit(close(con), add = TRUE)

		write(ls(object), file = con, sep = "\n")	
	}

	# print.txt
	writePrint <- function() {
		con <- createConnection("print.txt")
		sink(file = con)
		on.exit(sink(), add = TRUE)
		on.exit(close(con), add = TRUE)
		
		print(object)
	}

	# plot
	writePlot <- function() {
		png(paste(folder, "/plot%03d.png", sep = ""))
		on.exit(dev.off())

		plot(object)
	}

	classes <- class(object)
	firstClass <- classes[1]
	vectorClasses <- c("logical", "numeric", "integer", "character", 
		"factor", "Date", "POSIXct", "POSIXlt","POSIXt")

	if (firstClass %in% c(vectorClasses, "matrix", "data.frame")) {
		writeSummary()
		writeHead()
		writeDim()
	} else if (firstClass == "list") {
		writeSummary()
	} else if (firstClass == "environment") {
		writeLs()
	} else if (firstClass == "function") {
		writePrint()
	} else {
		# write for undefined class
		
		functions <- list(
			summary = writeSummary,
			plot = writePlot,
			print = writePrint
		)

		for (name in names(functions)) {
			for (cls in class(object)) {
				if (!is.null(getS3method(name, cls, optional = TRUE))) {
					do.call(functions[[name]], list())
					functions[[name]] <- NULL
					break
				}
			}
		}

		# output at least 'summary' for list
		if (typeof(object) == "list" && !is.null(functions$summary)) {
			writeSummary()
		}
	}
	
}

# import R Data 'file' into R object named 'name'
.importObject <- function(file, name) {
	env <- new.env()
	load(file, envir = env)
	objectName <- ls(env)[[1]]
	assign(name, env[[objectName]], envir = .GlobalEnv)
}

# create dummy variables
.createDummy <- function(data, column, value = c(1L, 0L), targets = NULL) {
	n <- nrow(data)
	
	if (is.null(targets)) {
		cl <- as.factor(data[[column]])
		labels <- levels(cl)
		x <- matrix(value[2], n, length(labels))
		x[(1L:n) + n * (unclass(cl) - 1L)] <- value[1]
		
		columnNames <- character()
		for (i in seq_along(labels)) {
			columnNames[i] <- paste(column, "_", labels[i], sep = "")
		}
		dimnames(x) <- list(NULL, columnNames)
		return(x)
	} else {
		labels <- targets
		x <- matrix(value[2], n, length(labels))
		columnNames <- character()
		for (i in seq_along(labels)) {
			x[,i][data[[column]] == labels[i]] <- value[1]
			columnNames[i] <- paste(column, "_", labels[i], sep = "")
		}
		dimnames(x) <- list(NULL, columnNames)
		return(x)
	}
}

# plot, but don't change current device
.backgroundPlot <- function(plotExpr, deviceNum = 0, sample = 0, width = NULL, height = NULL) {
	
	# open new device
	RflowGD(width = width, height = height)
	
	# close device on exit
	on.exit(dev.off())

	# do plot
	evalq(plotExpr)
}

## Detect packages required in R script (x). If attach.only = TRUE,
## it only returns packages that will can be attached.
## 
## TODO: Replace if there's a better implementation. Consider:
##        - If library() is called inside a function, it is attached when the function is called
##        - If detach() or unloadNamespace() is called, the package is not required after cache

.detectPackages <- function(x, attach.only = FALSE) {
	# NOTE: It only detect the 1st argument of functions
	inner.fun <- function(x, attach.only = FALSE) {
		ret <- character(0)
		
		# Search inside {...} / (...)
		braces <- unlist(lapply(x, inherits, what = c("{", "(")))
		if(any(braces)) {
			for(i in which(braces)) {
				for(j in 2:length(x[[i]])) {
					ret <- c(ret, Recall(x = x[[i]][[j]], attach.only = attach.only))
				}
			}
		}
		
		# Search inside calls including function() {...}.
		# It is considered only when attach.only == FALSE
		# because the library may be used only in the function.
		if(!attach.only) {
			calls <- unlist(lapply(x, inherits, what = "call"))
			if(any(calls)) {
				for(i in which(calls)) {
					ret <- c(ret, Recall(x = x[[i]], attach.only = attach.only))
				}
			}
		}
		
		funs.attach <- c("library", "require", "attachNamespace")
		funs.no.attach <- c("loadNamespace", "requireNamespace")
		
		if(attach.only) {
			funs <- funs.attach
		} else {
			funs <- c(funs.attach, funs.no.attach)
		}
		
		if(inherits(x, "call")) {
			# check whether it is library/require
			if(is.name(x[[1]]) && as.character(x[[1]]) %in% funs) {
				
				# return the package name
				if(x[[1]] == "attachNamespace") {
					arg <- "ns"
				} else {
					arg <- "package"
				}
				
				if((is.name(x[[2]]) || is.character(x[[2]])) && (is.null(names(x)) || names(x)[2] %in% c("", arg))) {
					ret <- c(ret, as.character(x[[2]]))
				}
			}
		}
		return(ret)
	}
	
	try.result <- try({
		parse.result <- base::parse(text = x)
		res1 <- unlist(lapply(parse.result, inner.fun, attach.only = attach.only))
		if(attach.only) {
			res2 <- character(0)
		} else {
			res2 <- base::subset(utils::getParseData(parse.result), token == "SYMBOL_PACKAGE")[["text"]]
		}
	}, silent = TRUE)
	
	if(inherits(try.result, "try-error")) {
		character(0)
	} else {
		sort(unique(c(res1, res2)))
	}
}

# Check if user library should be made
# Check if library directory exists and writable.
# install.packages() does more complicated things on Windows, which we may need to mimic.
.getUserLibraryPathIfNeeded <- function() {
	lib <- .libPaths()[1L]
	if(dir.exists(lib) & file.access(lib, 2) == 0L) {
		return(NA)
	} else {
		return(unlist(strsplit(Sys.getenv("R_LIBS_USER"), .Platform$path.sep))[1L])
	}
}

# Create user library and update .libPaths()
.createUserLibrary <- function() {
	user <- unlist(strsplit(Sys.getenv("R_LIBS_USER"), .Platform$path.sep))[1L]
	dir.create(user, showWarnings = FALSE, recursive = TRUE)
	.libPaths(c(user, .libPaths()));
}

.rflowPlotNumericMatrix <- function(x) {
	# plot correlation matrix
	.corPlot <- function(x) {
		diag(x) <- x[upper.tri(x)] <- NA
		label <- colnames(x)
		x <- rbind(NA, x, NA)
		
		lattice::levelplot(x, at = seq(-1, 1, by = .02), col.regions = colorRampPalette(c("#3B9AB2", "#EEEEEE", "#F21A00")),
			scales = list(x = list(at = NULL), y = list(at = NULL)),
			par.settings = list(
				axis.line = list(col = NA),
				layout.heights = list(
					top.padding = 0, main.key.padding = 0, key.axis.padding = 0,
					axis.xlab.padding = 0, key.sub.padding = 0, key.sub.padding = 0,
					bottom.padding = 0
				), 
				layout.widths = list(
					left.padding = 0, key.ylab.padding = 0, ylab.axis.padding = 0,
					axis.key.padding = 0
				)
			),
			xlab = NULL, ylab = NULL, panel=function(x, y, ...){
				lattice::panel.levelplot(x, y, cex = .8, ...)
				lattice::panel.text((1:length(label)) + 1.5, 1:length(label), labels = label, pos = 2, cex = .8)
			}
		)
	}
	
	d <- dim(x)
	if(d[1] == d[2] && all(x == t(x), na.rm = TRUE) && all(x >= -1 & x <= 1, na.rm = TRUE) && all(diag(x) == 1, na.rm = TRUE)) {
		.corPlot(x)
	} else {
		lattice::levelplot(x, scales = list(x = list(rot = 90)), col.regions = colorRampPalette(c("#3B9AB2", "#EEEEEE", "#F21A00")))
	}

}